<?php
/**
 * news-base functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package news-base
 */


if (!function_exists('news_base_setup')) :
    /**
     * Sets up theme defaults and registers support for various WordPress features.
     *
     * Note that this function is hooked into the after_setup_theme hook, which
     * runs before the init hook. The init hook is too late for some features, such
     * as indicating support for post thumbnails.
     */
    function news_base_setup()
    {
        /*
         * Make theme available for translation.
         * Translations can be filed in the /languages/ directory.
         * If you're building a theme based on news-base, use a find and replace
         * to change 'news-base' to the name of your theme in all the template files.
         */
        load_theme_textdomain( 'news-base', get_template_directory() . '/languages' );


        // Add default posts and comments RSS feed links to head.
        add_theme_support('automatic-feed-links');

        /*
         * Let WordPress manage the document title.
         * By adding theme support, we declare that this theme does not use a
         * hard-coded <title> tag in the document head, and expect WordPress to
         * provide it for us.
         */
        add_theme_support('title-tag');

        /*
         * Enable support for custom logo.
         */
        add_theme_support('custom-logo', array(
            'header-text' => array('site-title', 'site-description'),
        ));
        /*
         * Enable support for Post Thumbnails on posts and pages.
         *
         * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
         */
        add_theme_support('post-thumbnails');
        add_image_size('news-base-1200-675', 1200, 675, true);
        add_image_size('news-base-600-335', 600, 335, true);


        // Set up the WordPress core custom header feature.
        add_theme_support('custom-header', apply_filters('news_base_custom_header_args', array(
            'width' => 1400,
            'height' => 380,
            'flex-height' => true,
            'header-text' => false,
            'default-text-color' => '000',
        )));

        // This theme uses wp_nav_menu() in one location.
        register_nav_menus(array(
            'primary' => esc_html__('Primary Menu', 'news-base'),
            'social' => esc_html__('Social Menu', 'news-base'),
            'footer' => esc_html__('Footer Menu', 'news-base'),
        ));

        /*
         * Switch default core markup for search form, comment form, and comments
         * to output valid HTML5.
         */
        add_theme_support('html5', array(
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
        ));

         /*
         * Enable support for Post Formats.
         *
         * See: https://codex.wordpress.org/Post_Formats
         */
        add_theme_support( 'post-formats', array(
            'image',
            'video',
            'quote',
            'gallery',
            'audio',
        ) );


        // Set up the WordPress core custom background feature.
        add_theme_support('custom-background', apply_filters('news_base_custom_background_args', array(
            'default-color' => 'ffffff',
            'default-image' => '',
        )));

        // Add theme support for selective refresh for widgets.
        add_theme_support('customize-selective-refresh-widgets');

        /**
         * Load Init for Hook files.
         */
        require get_template_directory() . '/inc/hooks/hooks-init.php';

    }
endif;
add_action('after_setup_theme', 'news_base_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function news_base_content_width()
{
    $GLOBALS['content_width'] = apply_filters('news_base_content_width', 640);
}

add_action('after_setup_theme', 'news_base_content_width', 0);

/**
 * function for google fonts
 */
if (!function_exists('news_base_fonts_url')) :

    /**
     * Return fonts URL.
     *
     * @since 1.0.0
     * @return string Fonts URL.
     */
    function news_base_fonts_url()
    {

        $fonts_url = '';
        $fonts = array();
        $subsets = 'latin,latin-ext';

        /* translators: If there are characters in your language that are not supported by Open Sans, translate this to 'off'. Do not translate into your own language. */
        if ('off' !== _x('on', 'Open Sans font: on or off', 'news-base')) {
            $fonts[] = 'Open+Sans:400,400i,600,600i,700,700i';
        }

        /* translators: If there are characters in your language that are not supported by Merriweather, translate this to 'off'. Do not translate into your own language. */
        if ('off' !== _x('on', 'Merriweather font: on or off', 'news-base')) {
            $fonts[] = 'Merriweather:300,300i,400,400i,700,700i';
        }

        /* translators: If there are characters in your language that are not supported by Libre Franklin, translate this to 'off'. Do not translate into your own language. */
        if ('off' !== _x('on', 'Libre Franklin font: on or off', 'news-base')) {
            $fonts[] = 'Libre+Franklin:400,400i,600,600i,700,700i';
        }

        if ($fonts) {
            $fonts_url = add_query_arg(array(
                'family' => urldecode(implode('|', $fonts)),
                'subset' => urldecode($subsets),
            ), '//fonts.googleapis.com/css');
        }
        return $fonts_url;
    }
endif;
/**
 * Enqueue scripts and styles.
 */
function news_base_scripts()
{
    $min = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';
    wp_enqueue_style('jquery-slick', get_template_directory_uri() . '/assets/libraries/slick/css/slick' . $min . '.css');
    wp_enqueue_style('ionicons', get_template_directory_uri() . '/assets/libraries/ionicons/css/ionicons' . $min . '.css');
    wp_enqueue_style('bootstrap', get_template_directory_uri() . '/assets/libraries/bootstrap/css/bootstrap' . $min . '.css');
    wp_enqueue_style('sidr-nav', get_template_directory_uri() . '/assets/libraries/sidr/css/jquery.sidr.css');
    wp_enqueue_style('magnific-popup', get_template_directory_uri().'/assets/libraries/magnific-popup/magnific-popup.css');
    wp_enqueue_style('news-base-style', get_stylesheet_uri());
    wp_add_inline_style('news-base-style', news_base_trigger_custom_css_action());

    $fonts_url = news_base_fonts_url();
    if (!empty($fonts_url)) {
        wp_enqueue_style('news-base-google-fonts', $fonts_url, array(), null);
    }

    wp_enqueue_script('news-base-skip-link-focus-fix', get_template_directory_uri() . '/assets/libraries/js/skip-link-focus-fix.js', array(), '20151215', true);

    wp_enqueue_script('jquery-slick', get_template_directory_uri() . '/assets/libraries/slick/js/slick' . $min . '.js', array('jquery'), '', true);
    wp_enqueue_script('jquery-bootstrap', get_template_directory_uri() . '/assets/libraries/bootstrap/js/bootstrap' . $min . '.js', array('jquery'), '', true);
    wp_enqueue_script('jquery-match-height', get_template_directory_uri() . '/assets/libraries/jquery-match-height/jquery.matchHeight' . $min . '.js', array('jquery'), '', true);
    wp_enqueue_script('jquery-sidr', get_template_directory_uri() . '/assets/libraries/sidr/js/jquery.sidr.min.js', array('jquery'), '', true);
    wp_enqueue_script('jquery-sticky-sdebar', get_template_directory_uri() . '/assets/libraries/theiaStickySidebar/theia-sticky-sidebar.min.js', array('jquery'), '', true);
    wp_enqueue_script('jquery-magnific-popup', get_template_directory_uri().'/assets/libraries/magnific-popup/jquery.magnific-popup'.$min.'.js', array('jquery'), '', true);
    wp_enqueue_script('news-base-script', get_template_directory_uri() . '/assets/libraries/custom/js/custom-script.js', array('jquery'), '', 1);

    if (is_singular() && comments_open() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
}

add_action('wp_enqueue_scripts', 'news_base_scripts');

/**
 * Enqueue admin scripts and styles.
 */
function news_base_admin_scripts($hook)
{   
    wp_enqueue_style('news-base-admin-css', get_template_directory_uri() . '/assets/libraries/custom/css/admin.css');
    if ('widgets.php' === $hook) {
        wp_enqueue_media();
        wp_enqueue_script('news-base-custom-widgets', get_template_directory_uri() . '/assets/libraries/custom/js/widgets.js', array('jquery'), '1.0.0', true);
        
    }

}

add_action('admin_enqueue_scripts', 'news_base_admin_scripts');

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
require get_template_directory() . '/inc/jetpack.php';

/**
 * Customizer control scripts and styles.
 *
 * @since 1.0.5
 */
function news_base_customizer_control_scripts()
{

    wp_enqueue_style('news-base-customize-controls', get_template_directory_uri() . '/assets/libraries/custom/css/customize-controls.css');

}

add_action('customize_controls_enqueue_scripts', 'news_base_customizer_control_scripts', 0);
/**
 * Add featured image as background image to post navigation elements.
 *
 * @since news-base 1.0
 *
 */
function news_base_post_nav_background() {
    if ( ! is_single() ) {
        return;
    }

    $previous = ( is_attachment() ) ? get_post( get_post()->post_parent ) : get_adjacent_post( false, '', true );
    $next     = get_adjacent_post( false, '', false );
    $css      = '';

    if ( is_attachment() && 'attachment' == $previous->post_type ) {
        return;
    }

    if ( $previous &&  has_post_thumbnail( $previous->ID ) ) {
        $prevthumb = wp_get_attachment_image_src( get_post_thumbnail_id( $previous->ID ), 'post-thumbnail' );
        $css .= '
            .post-navigation .nav-previous { background-image: url(' . esc_url( $prevthumb[0] ) . '); }
            .post-navigation .nav-previous .post-title, .post-navigation .nav-previous a:hover .post-title, .post-navigation .nav-previous .meta-nav { color: #fff; }
            .post-navigation .nav-previous a:before { background-color: rgba(0, 0, 0, 0.4); }
        ';
    }

    if ( $next && has_post_thumbnail( $next->ID ) ) {
        $nextthumb = wp_get_attachment_image_src( get_post_thumbnail_id( $next->ID ), 'post-thumbnail' );
        $css .= '
            .post-navigation .nav-next { background-image: url(' . esc_url( $nextthumb[0] ) . '); border-top: 0; }
            .post-navigation .nav-next .post-title, .post-navigation .nav-next a:hover .post-title, .post-navigation .nav-next .meta-nav { color: #fff; }
            .post-navigation .nav-next a:before { background-color: rgba(0, 0, 0, 0.4); }
        ';
    }

    wp_add_inline_style( 'news-base-style', $css );
}
add_action( 'wp_enqueue_scripts', 'news_base_post_nav_background' );

/**
 * Descriptions on Primary Menu
 */
function news_base_header_menu_desc( $item_output, $item, $depth, $args ) {

    if( 'primary' == $args->theme_location  && $item->description )
        $item_output = str_replace( '</a>', '<span class="menu-description">' . $item->description . '</span></a>', $item_output );

    return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'news_base_header_menu_desc', 10, 4 );

function news_base_menu_notitle( $menu ){
    return $menu = preg_replace('/ title=\"(.*?)\"/', '', $menu );

}
add_filter( 'wp_nav_menu_items', 'news_base_custom_menu_item', 10, 2 );
function news_base_custom_menu_item ( $items, $args ) {
    if ($args->theme_location == 'primary') {
        $items = '<li class="base-address"><a href='.home_url() .'><span class="ion-ios-home"></span></a></li>'.$items;
    }
    return $items;
}
add_filter( 'wp_nav_menu', 'news_base_menu_notitle' );
add_filter( 'wp_page_menu', 'news_base_menu_notitle' );
add_filter( 'wp_list_categories', 'news_base_menu_notitle' );

