<?php 

/**
* Display default page as navigation if no custom menu was set
 */
if ( ! function_exists( 'definite_lite_fallback_menu' ) ) :
	function definite_lite_fallback_menu() {
		$pages = wp_list_pages( 'title_li=&echo=0' );
		echo '<ul id="menu-main-navigation" class="main-navigation-menu menu">' .  $pages  . '</ul>';
	}
endif;

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function definite_lite_body_classes( $classes ) {
		
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}		
	
	// Full post thumbnail disabled
	if ( is_single() && true == esc_attr(get_theme_mod( 'definite_lite_show_single_image', false ) ) ) {	
		$classes[] = 'hide-thumbnail';
	}
	
	// Full width site
	if ( true == esc_attr(get_theme_mod( 'definite_lite_show_fullwidth', false ) ) ) {	
		$classes[] = 'full-width';
	}

	if(basename(get_page_template()) === 'page.php' && ! is_single()){
		$classes[] = 'default-page';
	}
	
	// Check whether the current page is the full width short template
	if ( is_page_template( array( 'templates/template-short.php' ) ) ) {
		$classes[] = 'template-short';
	}		
	
	// Check whether the current page is the left column template
	if ( is_page_template( array( 'templates/template-left.php' ) ) ) {
		$classes[] = 'template-left';
	}

	// Check whether the current page is the right column template
	if ( is_page_template( array( 'templates/template-right.php' ) ) ) {
		$classes[] = 'template-right';
	}

	// Blog layout
	$definite_lite_blog_layout = esc_attr(get_theme_mod( 'definite_lite_blog_layout', 'classic' ) );	
	if( $definite_lite_blog_layout !== 'default' && ! is_singular() ) {
		$classes[] = 'blog-' . $definite_lite_blog_layout;
	}		
	
	// Single layout
	$definite_lite_single_layout = esc_attr(get_theme_mod( 'definite_lite_single_layout', 'right' ) );	
	if( $definite_lite_single_layout !== 'default' && is_single() ) {
		$classes[] = 'single-' . $definite_lite_single_layout;
	}	


	// Check whether we're in the customizer preview
	if ( ! has_post_thumbnail() ) {
		$classes[] = 'no-thumbnail';
	} else { 
		$classes[] = 'has-thumbnail';
	}
	
	if ( is_active_sidebar( 'breadcrumbs' ) && has_post_thumbnail() ) {
		$classes[] = 'has-breadcrumbs';
	}	

	return $classes;
}
add_filter( 'body_class', 'definite_lite_body_classes' );


/**
 * Adds custom classes to the array of post classes.
 *
 * @param array $classes Classes for the post element.
 * @return array
 */
function definite_lite_post_classes( $classes ) {

	// Add comments-off class.
	if ( ! ( comments_open() || get_comments_number() ) ) {
		$classes[] = 'comments-off';
	}

	return $classes;
}
add_filter( 'post_class', 'definite_lite_post_classes' );


//	Move the read more link outside of the post summary paragraph	
add_filter( 'the_content_more_link', 'definite_lite_move_more_link' );
	function definite_lite_move_more_link() {
		$definite_lite_read_more_text = esc_html( get_theme_mod( 'definite_lite_read_more_text' ) );
	return '<p><a class="readmore" href="'. esc_url(get_permalink()) . '">' . esc_html__( 'Read More', 'definite-lite' ) . '</a></p>';
}



/**
 * Change excerpt length for default posts
 * @param int $length Length of excerpt in number of words.
 * @return int
 */
function definite_lite_excerpt_length( $length ) {

	if ( is_admin() ) {
		return $length;
	}

	// Get excerpt length from database.
	$excerpt_length = esc_attr(get_theme_mod( 'definite_lite_excerpt_length', '35' ) );
	// Return excerpt text.
	if ( $excerpt_length >= 0 ) :
		return absint( $excerpt_length );
	else :
		return 35; // Number of words.
	endif;
}
add_filter( 'excerpt_length', 'definite_lite_excerpt_length' );


/**
 * Change excerpt more text for posts
 * @param String $more_text Excerpt More Text.
 * @return string
 */
function definite_lite_excerpt_more( $more_text ) {
	
	if ( is_admin() ) {
		return $more_text;
	}
	
	return '&hellip;<p><a class="readmore" href="'. esc_url(get_permalink()) . '">' . esc_html__( 'Read More', 'definite-lite' ) . '</a></p>';
}
add_filter( 'excerpt_more', 'definite_lite_excerpt_more' );

/**
 * Adding the read more button to our custom excerpt
 * @link https://codex.wordpress.org/Function_Reference/has_excerpt
 */
function definite_lite_excerpt_more_for_manual_excerpts( $excerpt ) {
    global $post;

    if ( has_excerpt( $post->ID ) ) {
		$excerpt_length = esc_attr(get_theme_mod( 'definite_lite_excerpt_length', '35' ) );
        $excerpt .= definite_lite_excerpt_more( '' );
    }

    return $excerpt;
}
add_filter( 'get_the_excerpt', 'definite_lite_excerpt_more_for_manual_excerpts' );



// Display the related posts
if ( ! function_exists( 'definite_lite_related_posts' ) ) {

   function definite_lite_related_posts() {
      wp_reset_postdata();
      global $post;

      // Define shared post arguments
      $args = array(
         'no_found_rows'            => true,
         'update_post_meta_cache'   => false,
         'update_post_term_cache'   => false,
         'ignore_sticky_posts'      => 1,
         'orderby'               => 'rand',
         'post__not_in'          => array($post->ID),
         'posts_per_page'        => 2
      );
      // Related by categories
      if ( get_theme_mod('definite_lite_related_posts', 'categories') == 'categories' ) {

         $cats = get_post_meta($post->ID, 'related-posts', true);

         if ( !$cats ) {
            $cats = wp_get_post_categories($post->ID, array('fields'=>'ids'));
            $args['category__in'] = $cats;
         } else {
            $args['cat'] = $cats;
         }
      }
      // Related by tags
      if ( get_theme_mod('definite_lite_related_posts', 'categories') == 'tags' ) {

         $tags = get_post_meta($post->ID, 'related-posts', true);

         if ( !$tags ) {
            $tags = wp_get_post_tags($post->ID, array('fields'=>'ids'));
            $args['tag__in'] = $tags;
         } else {
            $args['tag_slug__in'] = explode(',', $tags);
         }
         if ( !$tags ) { $break = true; }
      }

      $query = !isset($break)?new WP_Query($args):new WP_Query;
      return $query;
   }

}

/*	-----------------------------------------------------------------------------------------------
	IS COMMENT BY POST AUTHOR?
	Check if the specified comment is written by the author.
--------------------------------------------------------------------------------------------------- */

if ( ! function_exists( 'definite_lite_is_comment_by_post_author' ) ) :
	function definite_lite_is_comment_by_post_author( $comment = null ) {

		if ( is_object( $comment ) && $comment->user_id > 0 ) {
			$user = get_userdata( $comment->user_id );
			$post = get_post( $comment->comment_post_ID );
			if ( ! empty( $user ) && ! empty( $post ) ) {
				return $comment->user_id === $post->post_author;
			}
		}
		return false;

	}
endif;


/*	-----------------------------------------------------------------------------------------------
	COMMENT PLACEHOLDER TEXT
	Change comment form labels to become placeholders instead
--------------------------------------------------------------------------------------------------- */
function definite_lite_comment_textarea_placeholder( $args ) {
	$args['comment_field']        = str_replace( 'textarea', 'textarea placeholder="comment"', $args['comment_field'] );
	return $args;
}
add_filter( 'comment_form_defaults', 'definite_lite_comment_textarea_placeholder' );

// Comment Form Fields Placeholder
function definite_lite_comment_form_fields( $fields ) {
	foreach( $fields as &$field ) {
		$field = str_replace( 'id="author"', 'id="author" placeholder="name*"', $field );
		$field = str_replace( 'id="email"', 'id="email" placeholder="email*"', $field );
		$field = str_replace( 'id="url"', 'id="url" placeholder="website"', $field );
	}
	return $fields;
}
add_filter( 'comment_form_default_fields', 'definite_lite_comment_form_fields' );


/*	-----------------------------------------------------------------------------------------------
	CLASSIC SHORTCODE GALLERY
	Fix the default WP Shortcode gallery due to the incorrect structure
--------------------------------------------------------------------------------------------------- */

// http://techmagics.com/?p=2359
add_filter( 'post_gallery', 'definite_lite_post_gallery', 10, 2 );
function definite_lite_post_gallery( $output, $attr) {
    global $post, $wp_locale;

    static $instance = 0;
    $instance++;

    // We're trusting author input, so let's at least make sure it looks like a valid orderby statement
    if ( isset( $attr['orderby'] ) ) {
        $attr['orderby'] = sanitize_sql_orderby( $attr['orderby'] );
        if ( !$attr['orderby'] )
            unset( $attr['orderby'] );
    }

    extract(shortcode_atts(array(
        'order'      => 'ASC',
        'orderby'    => 'menu_order ID',
        'id'         => $post->ID,
        'itemtag'    => 'figure',
        'icontag'    => 'div',
        'captiontag' => 'figcaption',
        'columns'    => 3,
        'size'       => 'thumbnail',
        'include'    => '',
        'exclude'    => ''
    ), $attr));

    $id = intval($id);
    if ( 'RAND' == $order )
        $orderby = 'none';

    if ( !empty($include) ) {
        $include = preg_replace( '/[^0-9,]+/', '', $include );
        $_attachments = get_posts( array('include' => $include, 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $order, 'orderby' => $orderby) );

        $attachments = array();
        foreach ( $_attachments as $key => $val ) {
            $attachments[$val->ID] = $_attachments[$key];
        }
    } elseif ( !empty($exclude) ) {
        $exclude = preg_replace( '/[^0-9,]+/', '', $exclude );
        $attachments = get_children( array('post_parent' => $id, 'exclude' => $exclude, 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $order, 'orderby' => $orderby) );
		
   } else {
        $attachments = get_children( array('post_parent' => $id, 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $order, 'orderby' => $orderby) );
    }

    if ( empty($attachments) )
        return '';

    if ( is_feed() ) {
        $output = "\n";
        foreach ( $attachments as $att_id => $attachment )
            $output .= wp_get_attachment_link($att_id, $size, true) . "\n";
        return $output;
    }

    $itemtag = tag_escape($itemtag);
    $captiontag = tag_escape($captiontag);
    $columns = intval($columns);
    $itemwidth = $columns > 0 ? floor(100/$columns) : 100;
    $float = is_rtl() ? 'right' : 'left';

    $selector = "gallery-{$instance}";

    $output = apply_filters('definite_lite_post_gallery', "

        <!-- see gallery_shortcode() in wp-includes/media.php -->
        <div id='$selector' class='gallery galleryid-{$id} gallery-columns-{$columns} '>");

    $i = 0;
    foreach ( $attachments as $id => $attachment ) {

		$link = isset($attr['link']) && 'file' == $attr['link'] ? wp_get_attachment_link($id, $size, false, false) : wp_get_attachment_link($id, $size, true, false);

		$large_image = wp_get_attachment_image_src( $id, 'full' );
		$tn_image = wp_get_attachment_image_src( $id, 'thumbnail' );
		
    $output .= "<{$itemtag} class='gallery-item'>";
    $output .= "<{$icontag} class='gallery-icon'>";
    $output .= "<a href='{$large_image[0]}'>";
    $output .= "<img src='{$tn_image[0]}' alt='{$attachment->post_title}' />";
    if ( $captiontag && trim($attachment->post_excerpt) ) {
        $output .= "
            <{$captiontag} class='wp-caption-text gallery-caption'>
            " . wptexturize($attachment->post_excerpt) . "
            </{$captiontag}>";
    }
    $output .= "</a>";
    $output .= "</{$icontag}>";
    $output .= "</{$itemtag}>";
    }
	$output .= "</div>";
	
    return $output;
}


/*	-----------------------------------------------------------------------------------------------
	ATTACHMENT PAGE IMAGE NAVIGATION
	Added button classes to the image navigation links
--------------------------------------------------------------------------------------------------- */
add_filter( 'previous_image_link', 'definite_lite_img_link_class' );
add_filter( 'next_image_link',     'definite_lite_img_link_class' );


function definite_lite_img_link_class( $link )
{
    $class = 'next_image_link' === current_filter() ? 'next btn btn-primary' : 'prev btn btn-primary';

    return str_replace( '<a ', "<a class='$class'", $link );
}