<?php
/**
 * Custom template tags for this theme.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Zomer
 */

if ( ! function_exists( 'zomer_posted_on' ) ) :
	/**
	 * Prints HTML with meta information for the current post-date/time and author.
	 */
	function zomer_posted_on() {
		$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
		if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
			$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
		}

		$time_string = sprintf( $time_string,
			esc_attr( get_the_date( 'c' ) ),
			esc_html( get_the_date() ),
			esc_attr( get_the_modified_date( 'c' ) ),
			esc_html( get_the_modified_date() )
		);

		$posted_on = sprintf(
			esc_html_x( 'Posted on %s', 'post date', 'zomer' ),
			'<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $time_string . '</a>'
		);

		$byline = sprintf(
			esc_html_x( 'by %s', 'post author', 'zomer' ),
			'<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a></span>'
		);

		echo '<span class="posted-on">' . $posted_on . '</span><span class="byline"> ' . $byline . '</span>'; // WPCS: XSS OK.

	}
endif;

if ( ! function_exists( 'zomer_entry_archive_categories' ) ) :
	/**
	 * Prints HTML with meta information for the categories in archive views.
	 */
	function zomer_entry_archive_categories() {
		/* translators: used between list items, there is a space after the comma */
		$categories_list = get_the_category_list( esc_html__( ', ', 'zomer' ) );
		if ( $categories_list && zomer_categorized_blog() ) {
			printf( '<span class="cat-links">%s</span>', $categories_list ); // WPCS: XSS OK.
		}
	}
endif;

if ( ! function_exists( 'zomer_entry_footer' ) ) :
	/**
	 * Prints HTML with meta information for the categories, tags and comments.
	 */
	function zomer_entry_footer() {
		// Hide category and tag text for pages.
		if ( 'post' === get_post_type() ) {
			/* translators: used between list items, there is a space after the comma */
			$categories_list = get_the_category_list( esc_html__( ', ', 'zomer' ) );
			if ( $categories_list && zomer_categorized_blog() ) {
				printf( '<span class="cat-links secondary-font">' . esc_html__( 'Posted in %1$s', 'zomer' ) . '</span>', $categories_list ); // WPCS: XSS OK.
			}

			/* translators: used between list items, there is a space after the comma */
			$tags_list = get_the_tag_list( '', esc_html__( ', ', 'zomer' ) );
			if ( $tags_list ) {
				printf( '<span class="tags-links secondary-font">' . esc_html__( 'Tagged %1$s', 'zomer' ) . '</span>', $tags_list ); // WPCS: XSS OK.
			}
		}

		edit_post_link(
			sprintf(
				/* translators: %s: Name of current post */
				esc_html__( 'Edit %s', 'zomer' ),
				the_title( '<span class="screen-reader-text">"', '"</span>', false )
			),
			'<span class="edit-link">',
			'</span>'
		);
	}
endif;

/**
 * Returns true if a blog has more than 1 category.
 *
 * @return bool
 */
function zomer_categorized_blog() {
	if ( false === ( $all_the_cool_cats = get_transient( 'zomer_categories' ) ) ) {
		// Create an array of all the categories that are attached to posts.
		$all_the_cool_cats = get_categories( array(
			'fields'     => 'ids',
			'hide_empty' => 1,
			// We only need to know if there is more than one category.
			'number'     => 2,
		) );

		// Count the number of categories that are attached to the posts.
		$all_the_cool_cats = count( $all_the_cool_cats );

		set_transient( 'zomer_categories', $all_the_cool_cats );
	}

	if ( $all_the_cool_cats > 1 ) {
		// This blog has more than 1 category so zomer_categorized_blog should return true.
		return true;
	} else {
		// This blog has only 1 category so zomer_categorized_blog should return false.
		return false;
	}
}

/**
 * Flush out the transients used in zomer_categorized_blog.
 */
function zomer_category_transient_flusher() {
	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		return;
	}
	// Like, beat it. Dig?
	delete_transient( 'zomer_categories' );
}
add_action( 'edit_category', 'zomer_category_transient_flusher' );
add_action( 'save_post', 'zomer_category_transient_flusher' );

if ( ! function_exists( 'zomer_posted_on_archive' ) ) :
	/**
	 * Prints HTML with meta information for author and date.
	 */
	function zomer_posted_on_archive() {
		$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
		if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
			$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
		}

		$time_string = sprintf( $time_string,
			esc_attr( get_the_date( 'c' ) ),
			esc_html( get_the_date() ),
			esc_attr( get_the_modified_date( 'c' ) ),
			esc_html( get_the_modified_date() )
		);

		$byline = '<span class="by secondary-font"> ' . esc_html__( 'by', 'zomer' ) . ' </span><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a>';

		echo '<span class="byline">' . $byline . '</span><span class="posted-on secondary-font">' . $time_string . '</span>'; // WPCS: XSS OK.
	}
endif;

if ( ! function_exists( 'zomer_filter_archive_title' ) ) :
	/**
	 * Wrap archive's titles with a span tag.
	 *
	 * It makes much easier for user to hide archive's titles such as
	 * "Category:" or "Tag:".
	 */
	function zomer_filter_archive_title( $title ) {

		if ( is_category() ) {
			$title = sprintf( '<span>' . esc_html__( 'Category:', 'zomer' ) . '</span> %s', single_cat_title( '', false ) );
		} elseif ( is_tag() ) {
			$title = sprintf( '<span>' . esc_html__( 'Tag:', 'zomer' ) . '</span> %s', single_tag_title( '', false ) );
		} elseif ( is_post_type_archive() ) {
			$title = sprintf( '<span>' . esc_html__( 'Archives:', 'zomer' ) . '</span> %s', post_type_archive_title( '', false ) );
		}

		return $title;
	}
endif;
add_filter( 'get_the_archive_title', 'zomer_filter_archive_title' );

if ( ! function_exists( 'zomer_the_custom_logo' ) ) :
	/**
	 * Displays the optional custom logo.
	 *
	 * Does nothing if the custom logo is not available.
	 */
	function zomer_the_custom_logo() {
		if ( ! function_exists( 'the_custom_logo' ) ) {
			return;
		}

		the_custom_logo();
	}
endif;

/**
 * Displays footer credits.
 */
function zomer_set_footer_credits() {
    $credits = '<a href="' . esc_url( __( 'https://wordpress.org/', 'zomer' ) ) . '">' . sprintf( esc_html__( 'Proudly powered by %s', 'zomer' ), 'WordPress' ) . '</a>';
    $credits .= '<span class="sep"> | </span>';
    $credits .= sprintf( esc_html__( 'Theme: %1$s by %2$s.', 'zomer' ), 'Zomer', '<a href="https://themesharbor.com/" rel="designer">Themes Harbor</a>' );

    printf( '<div class="site-info">%s</div>', wp_kses_post( apply_filters( 'zomer_footer_credits', $credits ) ) );
}
add_action( 'zomer_footer_bottom', 'zomer_set_footer_credits' );

if ( ! function_exists( 'wp_body_open' ) ) :
	/**
	 * Shim for sites older than 5.2.
	 * Trigger wp_body_open action immediately after the opening body tag.
	 */
	function wp_body_open() {
		do_action( 'wp_body_open' );
	}
endif;

/**
 * Return early if Author Bio is not available.
 */
function zomer_author_bio() {
	if ( function_exists( 'jetpack_author_bio' ) ) {
		jetpack_author_bio();

		return;
	}

	get_template_part( 'template-parts/single/author', 'section' );
}
